import asyncio
import sys

from py_pli.pylib import VUnits, GlobalVar
from predefined_tasks.common.helper import send_to_gc
from virtualunits.HAL import HAL
from virtualunits.vu_node_application import VUNodeApplication
from virtualunits.vu_scan_table import VUScanTable
from virtualunits.vu_upper_measure_head import VUUpperMeasureHead

import config_enum.mover_enum as mover_config
import config_enum.scan_table_enum as st_config
import config_enum.upper_measure_head_enum as umh_config

hal_unit: HAL = VUnits.instance.hal
fmb_unit: VUNodeApplication = hal_unit.nodes['Mainboard']

node_unit = {
    'st'    : hal_unit.nodes['EEFNode'],
    'umh'   : hal_unit.nodes['EEFNode'],
    'excmc' : hal_unit.nodes['MC6'],
    'emsmc' : hal_unit.nodes['MC6'],
    'excls' : hal_unit.nodes['MC6'],
    'emsls' : hal_unit.nodes['MC6'],
    'xflfw' : hal_unit.nodes['MC6'],
    'pmtfw' : hal_unit.nodes['MC6'],
}

st_unit: VUScanTable = hal_unit.scan_table
umh_unit: VUUpperMeasureHead = hal_unit.upperMeasureHead

single_mover = {
    'excmc' : hal_unit.monochromatorExcitationMover,
    'emsmc' : hal_unit.monochromatorEmissionMover,
    'excls' : hal_unit.lightSwitchExcitation,
    'emsls' : hal_unit.lightSwitchEmission,
    'xflfw' : hal_unit.filterWheelFlashLamp,
    'pmtfw' : hal_unit.filterWheelPMT,
}

mover_name = {
    'st'    : 'Scan Table',
    'umh'   : 'Upper Measurement Head Mover',
    'excmc' : 'Excitation Monochromator Mover',
    'emsmc' : 'EmissionMonochromator Mover',
    'excls' : 'Excitation Light Switch',
    'emsls' : 'Emission Light Switch',
    'xflfw' : 'Flash Lamp Filter Wheel',
    'pmtfw' : 'PMT Filter Wheel',
}

position_unit = {
    'st'    : 'mm',
    'umh'   : 'mm',
    'excmc' : 'deg',
    'emsmc' : 'deg',
    'excls' : 'deg',
    'emsls' : 'deg',
    'xflfw' : 'deg',
    'pmtfw' : 'deg',
}


# High Level Function to start in GC ###################################################################################

async def st_stutter_test(profile=1, cycles=1, min_step=20.0):
    """ Scan Table Stutter Test """
    test_name = sys._getframe().f_code.co_name
    result = await st_mover_stutter_test(profile=profile, min_step_x=min_step, min_step_y=min_step, iterations=cycles)
    
    return f"{test_name} Result: {result}"


async def umh_stutter_test(cycles=1, min_step=5.0):
    """ Upper Measurement Head Stutter Test """
    test_name = sys._getframe().f_code.co_name
    result = await umh_mover_stutter_test(min_step=min_step, iterations=cycles)
    
    return f"{test_name} Result: {result}"


async def excmc_stutter_test(profile=1, cycles=1, min_step=45.0):
    """ Excitation Monochromator Stutter Test """
    mover_id = 'excmc'
    test_name = sys._getframe().f_code.co_name
    result = await mover_stutter_test(mover_id, profile=profile, min_step=min_step, iterations=cycles)
    
    return f"{test_name} Result: {result}"


async def emsmc_stutter_test(profile=1, cycles=1, min_step=45.0):
    """ Excitation Monochromator Stutter Test """
    mover_id = 'emsmc'
    test_name = sys._getframe().f_code.co_name

    result = await mover_stutter_test(mover_id, profile=profile, min_step=min_step, iterations=cycles)
    
    return f"{test_name} Result: {result}"


async def excls_stutter_test(profile=1, cycles=1, min_step=22.5):
    """ Excitation Monochromator Stutter Test """
    mover_id = 'excls'
    test_name = sys._getframe().f_code.co_name

    result = await mover_stutter_test(mover_id, profile=profile, min_step=min_step, iterations=cycles)
    
    return f"{test_name} Result: {result}"


async def emsls_stutter_test(profile=1, cycles=1, min_step=22.5):
    """ Excitation Monochromator Stutter Test """
    mover_id = 'emsls'
    test_name = sys._getframe().f_code.co_name

    result = await mover_stutter_test(mover_id, profile=profile, min_step=min_step, iterations=cycles)
    
    return f"{test_name} Result: {result}"


async def xflfw_stutter_test(profile=1, cycles=1, min_step=45.0):
    """ Excitation Monochromator Stutter Test """
    mover_id = 'xflfw'
    test_name = sys._getframe().f_code.co_name

    result = await mover_stutter_test(mover_id, profile=profile, min_step=min_step, iterations=cycles)
    
    return f"{test_name} Result: {result}"


async def pmtfw_stutter_test(profile=1, cycles=1, min_step=45.0):
    """ Excitation Monochromator Stutter Test """
    mover_id = 'pmtfw'
    test_name = sys._getframe().f_code.co_name

    result = await mover_stutter_test(mover_id, profile=profile, min_step=min_step, iterations=cycles)
    
    return f"{test_name} Result: {result}"


# Basic Functions ######################################################################################################

async def mover_stutter_test(mover_id='excmc', profile=1, start_pos='', end_pos='', min_step='', iterations=1):
    """
    Stutter Test implementation for a single axis mover. Move 0.1, then 0.9 and then the remaining step length. 
    This step is repeated from start to end and then back to start. Next move from start to end and back to start in a signle move.
    The test sequence is reapeated for the given number of iterations. And the mover gets homed every 10 iterations and after the last iterations.
    If the homing deviation stays within the tolerance every time, then the test is passed.

    Args:
        mover_id:
            The abbreviation of the mover to test.
            'st'    : Scan Table
            'umh'   : Upper Measurement Head
            'excmc' : Excitation Monochromator
            'emsmc' : EmissionMonochromator
            'excls' : Excitation Light Switch
            'emsls' : Emission Light Switch
            'xflfw' : Flash Lamp Filter Wheel
            'pmtfw' : PMT Filter Wheel
        profile:
            The HandleId of the mover profile to use for the test.
        start_pos:
            The start position of the test. Or an empty string to use the Min position of the movers config.
        end_pos:
            The end position of the test. Or an empty string to use the Min position of the movers config.
        min_step:
            Divide start to end into an integer number of steps with this minimum step size. Or an empty string to divide into 5 steps.
        iterations:
            The number of iteratons the test sequence is repeted.
    """

    GlobalVar.set_stop_gc(False)

    test_name = mover_id + '_' + sys._getframe().f_code.co_name

    mover = single_mover[mover_id]
    node = node_unit[mover_id]

    start_pos = float(start_pos) if start_pos != '' else mover.get_config(mover_config.Positions.Min)
    end_pos = float(end_pos) if end_pos != '' else mover.get_config(mover_config.Positions.Max)
    min_step = float(min_step) if min_step != '' else abs(end_pos - start_pos) / 5
  
    distance = end_pos - start_pos
    step_count = int(distance / min_step)
    step_size = distance / step_count

    pos_unit = position_unit[mover_id]

    max_pos_error = mover.get_config(mover_config.Mover.MaxStepsError)

    await send_to_gc(f"Starting {test_name}...")

    try:
        await node.StartFirmware()
        await mover.InitializeDevice()
        await mover.UseProfile(profile)

        await send_to_gc(f"Mover              : {mover_name[mover_id]}")
        await send_to_gc(f"Profile            : {profile}")
        await send_to_gc(f"Distance           : {distance} {pos_unit}")
        await send_to_gc(f"Number of Steps    : {step_count}")
        await send_to_gc(f"Step Size          : {step_size} {pos_unit}")
        await send_to_gc(f"Max Position Error : {max_pos_error} µsteps")
        await send_to_gc(f" ")

        await send_to_gc(f"Homing...")
        await mover.Home()

        await send_to_gc(f"Running Test...")

        for i in range(1, (iterations + 1)):
            if iterations > 1:
                await send_to_gc(f"Iteration {i}")

            await mover.Move(start_pos)

            new_pos = start_pos

            for _ in range(step_count):
                if GlobalVar.get_stop_gc():
                    await send_to_gc(f"{test_name}: Stopped by user")
                    return 'CANCELLED'

                new_pos = new_pos + 0.1
                await mover.Move(new_pos)

                new_pos = new_pos + 0.9
                await mover.Move(new_pos)

                new_pos = new_pos + (step_size - 1)
                await mover.Move(new_pos)

            for _ in range(step_count):
                if GlobalVar.get_stop_gc():
                    await send_to_gc(f"{test_name}: Stopped by user")
                    return 'CANCELLED'

                new_pos = new_pos - 0.1
                await mover.Move(new_pos)

                new_pos = new_pos - 0.9
                await mover.Move(new_pos)

                new_pos = new_pos - (step_size - 1)
                await mover.Move(new_pos)

            if GlobalVar.get_stop_gc():
                await send_to_gc(f"{test_name}: Stopped by user")
                return 'CANCELLED'

            await mover.Move(start_pos)

            if GlobalVar.get_stop_gc():
                await send_to_gc(f"{test_name}: Stopped by user")
                return 'CANCELLED'

            await mover.Move(end_pos)

            if GlobalVar.get_stop_gc():
                await send_to_gc(f"{test_name}: Stopped by user")
                return 'CANCELLED'

            await mover.Move(start_pos)

            if ((i % 10) == 0) or (i == iterations):
                await send_to_gc(f"Homing...")
                pos_error = await mover.Home()
                if abs(pos_error) > max_pos_error:
                    await send_to_gc(f"Homing deviation too big: {pos_error} µsteps (Tolerance {max_pos_error} µsteps)", log=True, error=True)
                    return 'FAILED'
                else:
                    await send_to_gc(f"Homing deviation: {pos_error} µsteps is in tolerances")

    except Exception as ex:
        await send_to_gc(f"{test_name}: Stopped, Error occurred: {repr(ex)}", log=True, error=True)
        return 'FAILED'

    return 'PASSED'


async def st_mover_stutter_test(profile=1, start_pos_x='', end_pos_x='', min_step_x='', start_pos_y='', end_pos_y='', min_step_y='', iterations=1):
    """
    Stutter Test implementation for the scan table. The test first tests the X axis then the Y axis the same way single axis movers are tested.
    See mover_stutter_test() for the basic test description. For the homing deviation both X and Y must always be in the tolerances to pass.

    Args:
        profile:
            The HandleId of the mover profile to use for the test.
        start_pos_x:
            The start position for the X axis test. Or an empty string to use the Min position of the movers config.
        end_pos_x:
            The end position for the X axis test. Or an empty string to use the Min position of the movers config.
        min_step_x:
            Divide start to end into an integer number of steps with this minimum step size for the X axis test. Or an empty string to divide into 5 steps.
        start_pos_Y:
            The start position for the Y axis test. Or an empty string to use the Min position of the movers config.
        end_pos_y:
            The end position for the Y axis test. Or an empty string to use the Min position of the movers config.
        min_step_y:
            Divide start to end into an integer number of steps with this minimum step size for the Y axis test. Or an empty string to divide into 5 steps.
        iterations:
            The number of iteratons the test sequence is repeted.
    """

    GlobalVar.set_stop_gc(False)

    test_name = sys._getframe().f_code.co_name

    mover = st_unit
    node = node_unit['st']

    min_position_x, min_position_y = mover.get_config(st_config.Positions.Min)
    max_position_x, max_position_y = mover.get_config(st_config.Positions.Max)

    start_pos_x = float(start_pos_x) if start_pos_x != '' else min_position_x
    end_pos_x = float(end_pos_x) if end_pos_x != '' else max_position_x
    min_step_x = float(min_step_x) if min_step_x != '' else abs(end_pos_x - start_pos_x) / 5

    start_pos_y = float(start_pos_y) if start_pos_y != '' else min_position_y
    end_pos_y = float(end_pos_y) if end_pos_y != '' else max_position_y
    min_step_y = float(min_step_y) if min_step_y != '' else abs(end_pos_y - start_pos_y) / 5
  
    distance_x = end_pos_x - start_pos_x
    step_count_x = int(distance_x / min_step_x)
    step_size_x = distance_x / step_count_x

    distance_y = end_pos_y - start_pos_y
    step_count_y = int(distance_y / min_step_y)
    step_size_y = distance_y / step_count_y

    pos_unit = position_unit['st']

    max_pos_error = mover.get_config(st_config.CoreXY.MaxStepsError)

    try:
        await node.StartFirmware()
        await mover.InitializeDevice()
        await mover.UseProfile(profile)

        await send_to_gc(f"Mover              : {mover_name['st']}")
        await send_to_gc(f"Profile            : {profile}")
        await send_to_gc(f"Distance X         : {distance_x} {pos_unit}")
        await send_to_gc(f"Distance Y         : {distance_y} {pos_unit}")
        await send_to_gc(f"Number of Steps X  : {step_count_x}")
        await send_to_gc(f"Number of Steps Y  : {step_count_y}")
        await send_to_gc(f"Step Size X        : {step_size_x} {pos_unit}")
        await send_to_gc(f"Step Size Y        : {step_size_y} {pos_unit}")
        await send_to_gc(f"Max Position Error : {max_pos_error} µsteps")
        await send_to_gc(f" ")

        await send_to_gc(f"Homing...")
        await mover.Home()

        # Iterate X-Positions
        await send_to_gc(f"Running Test in X...")

        for i in range(1, (iterations + 1)):
            if iterations > 1:
                await send_to_gc(f"X Iteration {i}")

            await mover.Move(start_pos_x, start_pos_y)

            new_pos_x = start_pos_x

            for _ in range(step_count_x):
                if GlobalVar.get_stop_gc():
                    await send_to_gc(f"{test_name}: Stopped by user")
                    return 'CANCELLED'

                new_pos_x = new_pos_x + 0.1
                await mover.Move(new_pos_x, start_pos_y)

                new_pos_x = new_pos_x + 0.9
                await mover.Move(new_pos_x, start_pos_y)

                new_pos_x = new_pos_x + (step_size_x - 1)
                await mover.Move(new_pos_x, start_pos_y)

            for _ in range(step_count_x):
                if GlobalVar.get_stop_gc():
                    await send_to_gc(f"{test_name}: Stopped by user")
                    return 'CANCELLED'

                new_pos_x = new_pos_x - 0.1
                await mover.Move(new_pos_x, start_pos_y)

                new_pos_x = new_pos_x - 0.9
                await mover.Move(new_pos_x, start_pos_y)

                new_pos_x = new_pos_x - (step_size_x - 1)
                await mover.Move(new_pos_x, start_pos_y)

            if GlobalVar.get_stop_gc():
                await send_to_gc(f"{test_name}: Stopped by user")
                return 'CANCELLED'

            await mover.Move(start_pos_x, start_pos_y)

            if GlobalVar.get_stop_gc():
                await send_to_gc(f"{test_name}: Stopped by user")
                return 'CANCELLED'

            await mover.Move(end_pos_x, start_pos_y)

            if GlobalVar.get_stop_gc():
                await send_to_gc(f"{test_name}: Stopped by user")
                return 'CANCELLED'

            await mover.Move(start_pos_x, start_pos_y)

            if ((i % 10) == 0) or (i == iterations):
                await send_to_gc(f"Homing...")
                pos_error_x, pos_error_y = await mover.Home()
                if (abs(pos_error_x) > max_pos_error) or (abs(pos_error_y) > max_pos_error):
                    await send_to_gc(f"Homing deviation too big: X = {pos_error_x} µsteps, Y = {pos_error_y} µsteps (Tolerance {max_pos_error} µsteps)", log=True, error=True)
                    return 'FAILED'
                else:
                    await send_to_gc(f"Homing deviation: X = {pos_error_x} µsteps, Y = {pos_error_y} µsteps is in tolerances")

        # Iterate Y-Positions
        await send_to_gc(f"Running Test in Y...")

        for i in range(1, (iterations + 1)):
            if iterations > 1:
                await send_to_gc(f"Y Iteration {i}")

            await mover.Move(start_pos_x, start_pos_y)

            new_pos_y = start_pos_y

            for _ in range(step_count_y):
                if GlobalVar.get_stop_gc():
                    await send_to_gc(f"{test_name}: Stopped by user")
                    return 'CANCELLED'

                new_pos_y = new_pos_y + 0.1
                await mover.Move(start_pos_x, new_pos_y)

                new_pos_y = new_pos_y + 0.9
                await mover.Move(start_pos_x, new_pos_y)

                new_pos_y = new_pos_y + (step_size_y - 1)
                await mover.Move(start_pos_x, new_pos_y)

            for _ in range(step_count_y):
                if GlobalVar.get_stop_gc():
                    await send_to_gc(f"{test_name}: Stopped by user")
                    return 'CANCELLED'

                new_pos_y = new_pos_y - 0.1
                await mover.Move(start_pos_x, new_pos_y)

                new_pos_y = new_pos_y - 0.9
                await mover.Move(start_pos_x, new_pos_y)

                new_pos_y = new_pos_y - (step_size_y - 1)
                await mover.Move(start_pos_x, new_pos_y)

            if GlobalVar.get_stop_gc():
                await send_to_gc(f"{test_name}: Stopped by user")
                return 'CANCELLED'

            await mover.Move(start_pos_x, start_pos_y)

            if GlobalVar.get_stop_gc():
                await send_to_gc(f"{test_name}: Stopped by user")
                return 'CANCELLED'

            await mover.Move(start_pos_x, end_pos_y)

            if GlobalVar.get_stop_gc():
                await send_to_gc(f"{test_name}: Stopped by user")
                return 'CANCELLED'

            await mover.Move(start_pos_x, start_pos_y)

            if ((i % 10) == 0) or (i == iterations):
                await send_to_gc(f"Homing...")
                pos_error_x, pos_error_y = await mover.Home()
                if (abs(pos_error_x) > max_pos_error) or (abs(pos_error_y) > max_pos_error):
                    await send_to_gc(f"Homing deviation too big: X = {pos_error_x} µsteps, Y = {pos_error_y} µsteps (Tolerance {max_pos_error} µsteps)", log=True, error=True)
                    return 'FAILED'
                else:
                    await send_to_gc(f"Homing deviation: X = {pos_error_x} µsteps, Y = {pos_error_y} µsteps is in tolerances")

    except Exception as ex:
        await send_to_gc(f"{test_name}: Stopped, Error occurred: {repr(ex)}", log=True, error=True)
        return 'FAILED'

    return 'PASSED'


async def umh_mover_stutter_test(start_pos='', end_pos='', min_step='', iterations=1):
    """
    Stutter Test implementation for the upper measurement head. The mover is tested the same way single axis movers are tested.
    See mover_stutter_test() for the basic test description.
    In addition to the homing deviation, also the tilt that is corrected when homing is checked and must not exceed the tolerances for the test to pass.

    Args:
        mover_id:
            The abbreviation of the mover to test.
            'st'    : Scan Table
            'umh'   : Upper Measurement Head
            'excmc' : Excitation Monochromator
            'emsmc' : EmissionMonochromator
            'excls' : Excitation Light Switch
            'emsls' : Emission Light Switch
            'xflfw' : Flash Lamp Filter Wheel
            'pmtfw' : PMT Filter Wheel
        profile:
            The HandleId of the mover profile to use for the test.
        start_pos:
            The start position of the test. Or an empty string to use the Min position of the movers config.
        end_pos:
            The end position of the test. Or an empty string to use the Min position of the movers config.
        min_step:
            Divide start to end into an integer number of steps with this minimum step size. Or an empty string to divide into 5 steps.
        iterations:
            The number of iteratons the test sequence is repeted.
    """

    GlobalVar.set_stop_gc(False)

    test_name = sys._getframe().f_code.co_name

    mover = umh_unit
    node = node_unit['umh']

    start_pos = float(start_pos) if start_pos != '' else mover.get_config(umh_config.Positions.Min)
    end_pos = float(end_pos) if end_pos != '' else mover.get_config(umh_config.Positions.Max)
    min_step = float(min_step) if min_step != '' else abs(end_pos - start_pos) / 5
  
    distance = end_pos - start_pos
    step_count = int(distance / min_step)
    step_size = distance / step_count

    pos_unit = position_unit['umh']

    max_pos_error = mover.get_config(mover_config.Mover.MaxStepsError)
    # max_tilt_error = 512

    await send_to_gc(f"Starting {test_name}...")

    try:
        await node.StartFirmware()
        await mover.InitializeDevice()

        await send_to_gc(f"Mover              : {mover_name['umh']}")
        await send_to_gc(f"Distance           : {distance} {pos_unit}")
        await send_to_gc(f"Number of Steps    : {step_count}")
        await send_to_gc(f"Step Size          : {step_size} {pos_unit}")
        await send_to_gc(f"Max Position Error : {max_pos_error} µsteps")
        # await send_to_gc(f"Max Tilt Error     : {max_tilt_error} µsteps")
        await send_to_gc(f" ")

        await send_to_gc(f"Homing...")
        await mover.Home()
        # await mover.home_with_tilt_correction()

        await send_to_gc(f"Running Test...")

        for i in range(1, (iterations + 1)):
            if iterations > 1:
                await send_to_gc(f"Iteration {i}")

            await mover.Move(start_pos)

            new_pos = start_pos

            for _ in range(step_count):
                if GlobalVar.get_stop_gc():
                    await send_to_gc(f"{test_name}: Stopped by user")
                    return 'CANCELLED'

                new_pos = new_pos + 0.1
                await mover.Move(new_pos)

                new_pos = new_pos + 0.9
                await mover.Move(new_pos)

                new_pos = new_pos + (step_size - 1)
                await mover.Move(new_pos)

            for _ in range(step_count):
                if GlobalVar.get_stop_gc():
                    await send_to_gc(f"{test_name}: Stopped by user")
                    return 'CANCELLED'

                new_pos = new_pos - 0.1
                await mover.Move(new_pos)

                new_pos = new_pos - 0.9
                await mover.Move(new_pos)

                new_pos = new_pos - (step_size - 1)
                await mover.Move(new_pos)

            if GlobalVar.get_stop_gc():
                await send_to_gc(f"{test_name}: Stopped by user")
                return 'CANCELLED'

            await mover.Move(start_pos)

            if GlobalVar.get_stop_gc():
                await send_to_gc(f"{test_name}: Stopped by user")
                return 'CANCELLED'

            await mover.Move(end_pos)

            if GlobalVar.get_stop_gc():
                await send_to_gc(f"{test_name}: Stopped by user")
                return 'CANCELLED'

            await mover.Move(start_pos)

            if ((i % 10) == 0) or (i == iterations):
                await send_to_gc(f"Homing...")
                pos_error = await mover.Home()
                # pos_error, tilt_error = await mover.home_with_tilt_correction()
                if abs(pos_error) > max_pos_error:
                    await send_to_gc(f"Homing deviation too big: {pos_error} µsteps (Tolerance {max_pos_error} µsteps)", log=True, error=True)
                    return 'FAILED'
                else:
                    await send_to_gc(f"Homing deviation: {pos_error} µsteps is in tolerances")
                # if abs(tilt_error) > max_tilt_error:
                #     await send_to_gc(f"Tilt deviation too big: {tilt_error} µsteps (Tolerance {max_tilt_error} µsteps)", log=True, error=True)
                #     return 'FAILED'
                # else:
                #     await send_to_gc(f"Tilt deviation: {tilt_error} µsteps is in tolerances")

    except Exception as ex:
        await send_to_gc(f"{test_name}: Stopped, Error occurred: {repr(ex)}", log=True, error=True)
        return 'FAILED'

    return 'PASSED'

